/**
 * \file: mspin_connection_adapter_iap2.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN Connection Adapter iAP2
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel ICT-ADITG/SW2 tfickel@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_appl_if_adapter.h"
#include "mspin_logging.h"
#include <mqueue.h>
#include <pthread.h>
#include <errno.h>
#include <string.h>
#include <sys/prctl.h>
#include <sys/stat.h>
#include <sys/time.h>

#define MSPIN_READ_MESSAGE_QUEUE "/mspinreadqueue"
//#define MSPIN_NATIVE_READ_BUFFER_SIZE 8192 //=> works, default size
//#define MSPIN_NATIVE_READ_BUFFER_SIZE 65536 //=> does not work
//#define MSPIN_NATIVE_READ_BUFFER_SIZE 32768 //=> works but max packet size on /dev/ffs/ep4 is 16kByte
#define MSPIN_NATIVE_READ_BUFFER_SIZE 16384 //=> works. This seems to be also the limit for /dev/ffs/ep4. But use with care

static mqd_t gReadQueue = -1;
static S32 gNativeWriteFD = -1;
static S32 gNativeReadFD = -1;
static BOOL gQuitReaderThread = FALSE;
static pthread_t gReaderThreadID = 0;
static pthread_mutex_t gWriteMutex;
static BOOL gCloseMessageQueue = FALSE;

static void mspin_conn_iap2_closeMessageQueue(mqd_t *pMQ, const char* name)
{
    S32 rc = 0;

    //Try to close and unlink message queue => no effect
    if (*pMQ >= 0)
    {
        rc = mq_close(*pMQ);
        if (0 == rc)
        {
            rc = mq_unlink(name);

            if (0 != rc )
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(%d, '%s') ERROR: Failed to unlink message queue with rc=%d (errno: '%s')",
                        __FUNCTION__, *pMQ, name, rc, strerror(errno));
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(%d, '%s') message queue closed", __FUNCTION__, *pMQ, name);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(%d, '%s') ERROR: Failed to close message queue with rc=%d (errno: '%s')",
                    __FUNCTION__, *pMQ, name, rc, strerror(errno));
        }

        *pMQ = -1;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(%d, '%s') No message queue to close", __FUNCTION__, *pMQ, name);
    }
}

static void* mspin_conn_iap2_readThread(void* exinf)
{
    U8 readbuf[MSPIN_NATIVE_READ_BUFFER_SIZE] = {0};
    S32 rc = 0;
    S32 sendResult = 0;

    //Set thread name
    rc = prctl(PR_SET_NAME,"mspin_reader",0,0,0);

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(exinf=%p) started", __FUNCTION__, exinf);

    //Set cancelability state and type
    rc = pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
    if (rc != 0)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(exinf=%p) pthread_setcancelstate() failed %d",
                __FUNCTION__, exinf, rc);
        //continue after while loop
    }

    gCloseMessageQueue = FALSE; //reset

    //Reader loop
    while (!gQuitReaderThread && (rc >= 0) && (gNativeReadFD > -1))
    {
        mspin_log_printLn(eMspinVerbosityVerbose,
                "%s(exinf=%p) call read", __FUNCTION__, exinf);
        rc = read(gNativeReadFD, (char*)readbuf, MSPIN_NATIVE_READ_BUFFER_SIZE);
        mspin_log_printLn(eMspinVerbosityVerbose,
                "%s(exinf=%p) read returned with rc=%d", __FUNCTION__, exinf, rc);
        if (rc >= 0)
        {
            sendResult = mq_send(gReadQueue, (char*)readbuf, rc, 0);
            if (sendResult >= 0)
            {
                mspin_log_printLn(eMspinVerbosityVerbose,
                        "%s(exinf=%p) %p with len=%d written into mqueue",
                        __FUNCTION__, exinf, readbuf, rc);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(exinf=%p) ERROR: Failed to write data with len=%d into queue with rc=%d -> terminate thread",
                        __FUNCTION__, exinf, rc, sendResult);
                rc = sendResult;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(exinf=%p) ERROR: Failed to read data with rc=%d (errno: '%s') -> terminate thread",
                    __FUNCTION__, exinf, rc, strerror(errno));
            //leave because rc is negative
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(exinf=%p) terminate", __FUNCTION__, exinf);

    //In order to close receiver send a zero byte packet to let him return and then it closes
    // message queue when it checks that 'gCloseMessageQueue' is true
    gCloseMessageQueue = TRUE;
    sendResult = mq_send(gReadQueue, (char*)readbuf, 0, 0);
    if (sendResult < 0)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(exinf=%p) ERROR: Failed to write zero byte packet into queue with rc=%d",
                __FUNCTION__, exinf, rc, sendResult);
        rc = sendResult;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(exinf=%p) zero byte packet sent", __FUNCTION__, exinf);
    }

    pthread_exit((void*)exinf);
    return NULL;
}

static S32 mspin_conn_iap2_createThread(pthread_t *thread, void *(*threadFunction) (void *),
        const char* thread_name, void* exinf)
{
    S32 rc = -1;
    pthread_attr_t attr;
    (void)thread_name;	//unused

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: Failed to init thread attributes", __FUNCTION__);
        return rc;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return rc;
    }

    rc = pthread_create(thread, &attr, threadFunction, exinf);
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                        "%s() thread id is %d", __FUNCTION__, *thread);
    }

    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: Setting name failed with rc=%d", __FUNCTION__, rc);
        *thread = 0;
    }

    (void)pthread_attr_destroy(&attr);

    return rc;
}

static MSPIN_ERROR mspin_conn_iap2_startReaderThread(void)
{
    struct mq_attr attr;
    S32 rc = 0;

    //Open message queue
    if (gReadQueue < 0)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() open read message queue", __FUNCTION__);

        //Set message queue parameter
        attr.mq_curmsgs = 0;
        attr.mq_flags = 0;
        attr.mq_maxmsg = 10;
        attr.mq_msgsize = MSPIN_NATIVE_READ_BUFFER_SIZE;

        gReadQueue = mq_open(MSPIN_READ_MESSAGE_QUEUE, O_RDWR|O_CREAT, S_IRWXU, &attr);
        if (gReadQueue < 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Failed to open read message queue with rc=%d (errno=%s)",
                    __FUNCTION__, gReadQueue, strerror(errno));
            return MSPIN_ERROR_CONNECTION_START;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s() read message queue %d opened",
                    __FUNCTION__, gReadQueue);
        }
    }

    //Start reader thread
    gQuitReaderThread = FALSE;
    rc = mspin_conn_iap2_createThread(&gReaderThreadID, &mspin_conn_iap2_readThread, "mspin_reader", NULL);
    if (0 != rc)
    {
        gReaderThreadID = 0;
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: Failed to create reader thread with code=%d",
                __FUNCTION__, rc);
        return MSPIN_ERROR_CONNECTION_START;
    }

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_conn_iap2_setupNativeTransportConnection(const char *pReadDeviceName, const char *pWriteDeviceName)
{
    //Check write device name
    if (!pWriteDeviceName)
    {
        gNativeWriteFD = -1;
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s() FATAL ERROR: Write device name is NULL", __FUNCTION__);
        return MSPIN_ERROR_CONNECTION_START;
    }

    //Check read device name
    if (!pReadDeviceName)
    {
        gNativeReadFD = -1;
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s() FATAL ERROR: Read device name is NULL", __FUNCTION__);
        return MSPIN_ERROR_CONNECTION_START;
    }

    //Initialize write mutex
    if(0 != pthread_mutex_init(&gWriteMutex, NULL))
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s() FATAL ERROR: Failed to init write mutex", __FUNCTION__);
        return MSPIN_ERROR_CONNECTION_START;
    }

    //Open write device
    gNativeWriteFD = open(pWriteDeviceName, O_NONBLOCK | O_WRONLY);
    if (gNativeWriteFD < 0)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(read='%s', write='%s') ERROR: Failed to open write device with %d",
                __FUNCTION__, pReadDeviceName, pWriteDeviceName, gNativeWriteFD);
        return MSPIN_ERROR_CONNECTION_START;
    }

    //Open read device
    gNativeReadFD = open(pReadDeviceName, O_RDONLY);
    if (gNativeReadFD < 0)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(read='%s', write='%s') ERROR: Failed to open read device with %d",
                __FUNCTION__, pReadDeviceName, pWriteDeviceName, gNativeReadFD);
        return MSPIN_ERROR_CONNECTION_START;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(read='%s', write='%s') devices opened -> start reader thread",
            __FUNCTION__, pReadDeviceName, pWriteDeviceName);

    return mspin_conn_iap2_startReaderThread();
}

void mspin_conn_iap2_shutdownConnection(void)
{
    void* status;

    mspin_log_printLn(eMspinVerbosityDebug, "%s()", __FUNCTION__);

    gQuitReaderThread = TRUE;

    //Join reader thread
    if (gReaderThreadID > 0)
    {
        (void)pthread_cancel(gReaderThreadID);
        (void)pthread_join(gReaderThreadID, &status);
        gReaderThreadID = 0;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s() reader thread joined", __FUNCTION__);

    //Clean read file descriptor and reader message queue
    if (gNativeReadFD >= 0)
    {
        close(gNativeReadFD);
        gNativeReadFD = -1;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s() native read fd closed", __FUNCTION__);

    mspin_conn_iap2_closeMessageQueue(&gReadQueue, MSPIN_READ_MESSAGE_QUEUE);

    if (gNativeWriteFD >= 0)
    {
        close(gNativeWriteFD);
        gNativeWriteFD = -1;
    }

    mspin_log_printLn(eMspinVerbosityInfo, "%s() done", __FUNCTION__);
}

S32 mspin_conn_iap2_send(const U8* buffer, U32 bufferLen)
{
    S32 rc = MSPIN_SUCCESS;

    if (gNativeWriteFD > 0)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(buffer=%p, len=%d) writing (endpoint address=%d)",
                __FUNCTION__, buffer, bufferLen, rc);

        if (0 != pthread_mutex_lock(&gWriteMutex))
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(buffer=%p, len=%d) FATAL ERROR: Failed to acquire mutex lock",
                    __FUNCTION__, buffer, bufferLen);

            /* PRQA: Lint Message 454: A thread mutex isn't locked in this case. So ignore this error */
            /*lint -save -e454*/
            return -1;
            /*lint -restore*/
        }

        rc = write(gNativeWriteFD, buffer, bufferLen);
        pthread_mutex_unlock(&gWriteMutex); //try to unlock write mutex

        if ((rc >= 0) && (bufferLen == (U32)rc))
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(buffer=%p, len=%d) written",
                    __FUNCTION__, buffer, bufferLen);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(buffer=%p, len=%d) ERROR: Failed to write with rc=%d ('%s') -> close fd=%d",
                    __FUNCTION__, buffer, bufferLen, rc, strerror(errno), gNativeWriteFD);

            if (gNativeWriteFD >= 0)
            {
                close(gNativeWriteFD);
                gNativeWriteFD = -1;
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(buffer=%p, len=%d) ERROR: Device for writing data isn't open",
                __FUNCTION__, buffer, bufferLen, rc);
        rc = -1;
    }

    return rc;
}

S32 mspin_conn_iap2_receive(U8* buffer, U32 bufferLen, U32 timeout)
{
    S32 rc = -1;

    //Read from message queue
    if (0 < gReadQueue)
    {
        struct timespec currentTime;
        struct timespec absTime;

        mspin_log_printLn(eMspinVerbosityVerbose, "%s(buffer=%p, len=%d, timeout=%d) wait for data",
                __FUNCTION__, buffer, bufferLen, timeout);

        //It would be better to use CLOCK_MONOTONIC for timers in order to avoid
        // problems when the system time is changed. But CLOCK_MONOTONIC can't be used for mq_timedreceive. See
        // http://stackoverflow.com/questions/33873374/pthreads-mq-timedreceive-pthread-cond-timedwait-and-clock-realtime
        // => but this would be only a problem if the time is set back and no data is put into the message queue.
        clock_gettime(CLOCK_REALTIME, &currentTime);

        absTime.tv_sec = currentTime.tv_sec + (timeout / 1000);
        absTime.tv_nsec = currentTime.tv_nsec + (timeout % 1000)*1000000;

        if (absTime.tv_nsec >= 1000000000)
        {
            absTime.tv_sec += 1;
            absTime.tv_nsec -= 1000000000;
        }

        //Check if nano seconds are bigger than 1000000000ns (=1s) (compare man page of mq_timedreceive)
        rc = (S32)mq_timedreceive(gReadQueue, (char*)buffer, bufferLen, NULL, &absTime);
        if (0 < rc)
        {
            mspin_log_printLn(eMspinVerbosityVerbose, "%s(buffer=%p, len=%d, timeout=%d) %d data read",
                    __FUNCTION__, buffer, bufferLen, timeout, rc);
        }
        else
        {
            if ((0 == rc) && gCloseMessageQueue)
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(buffer=%p, len=%d, timeout=%d) %d bytes read -> close message queue",
                        __FUNCTION__, buffer, bufferLen, timeout, rc);

                mspin_conn_iap2_closeMessageQueue(&gReadQueue, MSPIN_READ_MESSAGE_QUEUE);
                gCloseMessageQueue = FALSE;
            }
            else if ((-1 == rc) && (errno == ETIMEDOUT))
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(buffer=%p, len=%d, timeout=%d) timeout occurred => return 0",
                        __FUNCTION__, buffer, bufferLen, timeout);
                rc = 0;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(buffer=%p, len=%d, timeout=%d) ERROR: Reading data failed with rc=%d (errno='%s')",
                        __FUNCTION__, buffer, bufferLen, timeout, rc, strerror(errno));
            }
            //return return code of mq_receive
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(buffer=%p, len=%d, timeout=%d) ERROR: No open read message queue",
                __FUNCTION__, buffer, bufferLen, timeout);
        //return default error value
    }

    return rc;
}
